<?php 
/*
Plugin Name: MM Wortwolken
Description: Einfache *entim*ter-ähnliche Wortwolken über URL-Parameter (?code=XXXXXXXXXX) mit Admin-Export.
Author: ChatGPT and PBMod
Author URI: https://github.com/svenbolte/
License: Closed Source (Freeware)
License URI: license.txt im Plugin-Ordner
Tags: wordcloud, voting, interactive, presentation, brainstorming, workshop, ajax
Version: 1.3.8
Stable tag: 1.3.8
Requires at least: 6.0
Tested up to: 6.9
Requires PHP: 8.2
*/

// ----------------------------- MM Wordcloud (Tagcloud) ähnlich wie Menti -----------------------------

class MM_Wordcloud {
    private static $instance;
    private $table_clouds;
    private $table_words;

    private const CODE_RE = '/^[0-9a-fA-F]{10}$/';
    private const TAGCOLOR_STEPS = [-0.6, -0.4, -0.2, 0, 0.2, 0.4, 0.6, 0.8];

    public static function instance() {
        return self::$instance ??= new self();
    }

    private function __construct() {
        global $wpdb;
        $this->table_clouds = $wpdb->prefix . 'menti_clouds';
        $this->table_words  = $wpdb->prefix . 'menti_words';

        $this->maybe_upgrade_schema();

        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_init', [$this, 'handle_admin_actions']);

        add_shortcode('menti_cloud', [$this, 'shortcode_menti_cloud']);

        foreach (['menti_add_word' => 'ajax_add_word', 'menti_vote_word' => 'ajax_vote_word', 'menti_cloud_view' => 'ajax_cloud_view'] as $action => $method) {
            add_action("wp_ajax_$action", [$this, $method]);
            add_action("wp_ajax_nopriv_$action", [$this, $method]);
        }

        // Delete nur für eingeloggte Admins (kein nopriv)
        add_action('wp_ajax_menti_delete_word', [$this, 'ajax_delete_word']);
    }

    /** Aktivierung: Tabellen anlegen / aktualisieren */
    public static function activate() {
        global $wpdb;
        $table_clouds = $wpdb->prefix . 'menti_clouds';
        $table_words  = $wpdb->prefix . 'menti_words';
        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $sql1 = "CREATE TABLE $table_clouds (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            code CHAR(10) NOT NULL UNIQUE,
            title VARCHAR(255) NOT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        $sql2 = "CREATE TABLE $table_words (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            cloud_id BIGINT(20) UNSIGNED NOT NULL,
            word VARCHAR(255) NOT NULL,
            author CHAR(3) NOT NULL DEFAULT 'XXX',
            emoji VARCHAR(32) NOT NULL DEFAULT '❔',
            count INT(11) NOT NULL DEFAULT 1,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY cloud_word (cloud_id, word),
            KEY cloud_id (cloud_id)
        ) $charset_collate;";

        dbDelta($sql1);
        dbDelta($sql2);
    }

    /** Bei Updates fehlende Spalten nachrüsten */
    private function maybe_upgrade_schema() {
        global $wpdb;
        $table = $this->table_words;

        $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));
        if ($exists !== $table) return;

        $has_author = $wpdb->get_var("SHOW COLUMNS FROM {$table} LIKE 'author'");
        $has_emoji  = $wpdb->get_var("SHOW COLUMNS FROM {$table} LIKE 'emoji'");
        if (empty($has_author) || empty($has_emoji)) self::activate();
    }

    /** Admin-Menü in wpdoodlez
    public function register_admin_menu() {
        add_submenu_page(
            'edit.php?post_type=wpdoodle',
            __('MM wordclouds', 'WPdoodlez'),
            __('MM wordclouds', 'WPdoodlez'),
            'manage_options',
            'menti-clouds',
            [$this, 'render_admin_page']
        );
    }
	
 */	
	
    /**   * Admin-Menü Plugin **/
    public function register_admin_menu() {
        add_menu_page(
            'MM Wortwolken',
            'MM TagClouds',
            'manage_options',
            'menti-clouds',
            array( $this, 'render_admin_page' ),
            'dashicons-cloud',
            26
        );
    }


    /** Admin Aktionen (Add/Delete/Reset/Export) */
    public function handle_admin_actions() {
        if (!current_user_can('manage_options')) return;

        $action = $_POST['menti_action'] ?? '';
        if ($action) {
            switch ($action) {
                case 'add':
                    check_admin_referer('menti_add_cloud');
                    $title = isset($_POST['menti_title']) ? sanitize_text_field(wp_unslash($_POST['menti_title'])) : '';
                    if ($title !== '') $this->create_cloud($title);
                    break;

                case 'delete':
                    $id = intval($_POST['cloud_id'] ?? 0);
                    $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
                    if ($id > 0 && wp_verify_nonce($nonce, 'menti_delete_cloud_' . $id)) $this->delete_cloud($id);
                    break;

				case 'reset':
					$id = intval($_POST['cloud_id'] ?? 0);
					$nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
					if ($id > 0 && wp_verify_nonce($nonce, 'menti_reset_cloud_' . $id)) {
						$this->reset_cloud_words($id);
					}
					break;
            }
        }

        if (
            isset($_GET['page'], $_GET['menti_action'], $_GET['cloud_id'], $_GET['_wpnonce']) &&
            $_GET['page'] === 'menti-clouds' &&
            $_GET['menti_action'] === 'export'
        ) {
            $id = intval($_GET['cloud_id']);
            $nonce = sanitize_text_field(wp_unslash($_GET['_wpnonce']));
            if ($id > 0 && wp_verify_nonce($nonce, 'menti_export_cloud_' . $id)) {
                $this->export_cloud_csv($id);
                exit;
            }
        }
    }

    private function create_cloud($title) {
        global $wpdb;

        do {
            $code = function_exists('random_bytes')
                ? substr(bin2hex(random_bytes(5)), 0, 10)
                : substr(md5(uniqid('', true)), 0, 10);

            $code = strtolower($code);
            $exists = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_clouds} WHERE code = %s",
                $code
            ));
        } while ($exists);

        $wpdb->insert(
            $this->table_clouds,
            ['code' => $code, 'title' => $title],
            ['%s', '%s']
        );
    }

    private function delete_cloud($id) {
        global $wpdb;
        $wpdb->delete($this->table_words, ['cloud_id' => $id], ['%d']);
        $wpdb->delete($this->table_clouds, ['id' => $id], ['%d']);
    }

    private function reset_cloud_words($id) {
        global $wpdb;
        $wpdb->delete($this->table_words, ['cloud_id' => $id], ['%d']);
    }

    /** Anzahl der Wörter (Einträge) für eine Wortwolke */
    private function get_word_count_for_cloud($cloud_id) {
        global $wpdb;
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_words} WHERE cloud_id = %d",
            $cloud_id
        ));
    }

    private function export_cloud_csv($id) {
    if (!current_user_can('manage_options')) wp_die('Keine Berechtigung.');

    global $wpdb;
    $cloud = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$this->table_clouds} WHERE id = %d", $id));
    if (!$cloud) wp_die('Wortwolke nicht gefunden.');

    // Daten wie in der Admin-Tabelle
    $words = $this->get_words_for_cloud($cloud->id);
    $word_count = is_array($words) ? count($words) : 0;
    $total_mentions = $this->get_total_mentions_for_cloud($cloud->id);

    // Zeitraum ermitteln (wie in render_admin_page)
    $oldest_ts = null;
    $newest_ts = null;
    if (is_array($words)) {
        foreach ($words as $row) {
            $dt_value = $row['created_at'] ?? '';
            $ts = is_numeric($dt_value) ? (int)$dt_value : strtotime((string)$dt_value);
            if (!$ts) continue;

            if ($newest_ts === null || $ts > $newest_ts) $newest_ts = $ts;
            if ($oldest_ts === null || $ts < $oldest_ts) $oldest_ts = $ts;
        }
    }

    $oldest_str = '';
    $newest_str = '';
    $diff_days = '';
    if ($oldest_ts && $newest_ts) {
        $oldest_dt = new DateTime('@' . $oldest_ts);
        $newest_dt = new DateTime('@' . $newest_ts);
        $oldest_dt->setTimezone(wp_timezone());
        $newest_dt->setTimezone(wp_timezone());
        $diff = $oldest_dt->diff($newest_dt);

        $oldest_str = $oldest_dt->format('d.m.Y H:i');
        $newest_str = $newest_dt->format('d.m.Y H:i');
        $diff_days = (string) intval($diff->days);
    }

    nocache_headers();
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="menti_cloud_' . $cloud->code . '_tabelle.csv"');

    $out = fopen('php://output', 'w');
    // UTF-8 BOM für Excel
    fprintf($out, chr(0xEF) . chr(0xBB) . chr(0xBF));

    // Header wie in der Admin-Tabelle
    fputcsv($out, ['Datum/Zeit', 'Autor', 'Titel', 'Nennungen', 'Anteil (%)', 'Graph'], ';', escape: "");

    if (is_array($words)) {
        foreach ($words as $row) {
            $count = (int) ($row['count'] ?? 0);
            $percent = $total_mentions > 0 ? ($count / $total_mentions) * 100 : 0;

            $dt_value = $row['created_at'] ?? '';
            $dt_text = $this->format_datetime($dt_value);

            $author_text = trim(($row['emoji'] ?? '') . ' ' . ($row['author'] ?? ''));
            $title_text = $this->pccf($row['word'] ?? '');

            $percent_text = number_format($percent, 1, ',', '');
            $graph_text = $percent_text; // in der Tabelle ist es ein Progress-Balken; in CSV der Prozentwert

            fputcsv($out, [$dt_text, $author_text, $title_text, $count, $percent_text, $graph_text], ';', escape: "");
        }
    }

    // Footer-Werte (tfoot) mit ausgeben
fputcsv($out, [], ';', escape: ""); // Leerzeile

// Prozentwerte sammeln
$percents = [];
if (is_array($words)) {
    foreach ($words as $row) {
        $count = (int) ($row['count'] ?? 0);
        if ($total_mentions > 0) {
            $percents[] = ($count / $total_mentions) * 100;
        }
    }
}

$min_p = $percents ? min($percents) : 0;
$max_p = $percents ? max($percents) : 0;
$avg_p = $percents ? (array_sum($percents) / count($percents)) : 0;

$footer_parts = [
    'Gesamt',
    'Beiträge: ' . (string) $word_count,
    'Upvotes: ' . (string) intval($total_mentions),
    'Min %: ' . number_format($min_p, 1, ',', ''),
    'Max %: ' . number_format($max_p, 1, ',', ''),
    'Ø %: ' . number_format($avg_p, 1, ',', ''),
];

if ($oldest_str && $newest_str) {
    $footer_parts[] = 'Zeitraum: ' . $oldest_str . ' – ' . $newest_str;
    $footer_parts[] = 'Tage: ' . $diff_days;
}

fputcsv($out, $footer_parts, ';', escape: "");

    fclose($out);
    exit;
}

    /** Admin-Seite */
    public function render_admin_page() {
        if (!current_user_can('manage_options')) return;

        global $wpdb;
        $clouds = $wpdb->get_results("SELECT * FROM {$this->table_clouds} ORDER BY created_at DESC");
        ?>
        <div class="wrap">
            <h1>MM Wortwolken</h1>
            <p>Lege hier Wortwolken an. Nutze auf einer normalen Seite oder in einem Beitrag (Titel z.B. Wordclooud) den Shortcode <code>[menti_cloud]</code> 
                und rufe die Seite dann mit <code>?code=DEINCODE</code> auf.</p>
            <h2>Neue Wortwolke anlegen</h2>
            <form method="post">
                <?php wp_nonce_field('menti_add_cloud'); ?>
                <input type="hidden" name="menti_action" value="add" />
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="menti_title">Titel</label></th>
                        <td><input name="menti_title" type="text" id="menti_title" class="regular-text" required></td>
                    </tr>
                </table>
                <?php submit_button('Wortwolke anlegen'); ?>
            </form>
            <h2>Bestehende Wortwolken</h2>
            <?php if (!empty($clouds)) : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>Titel</th>
                            <th>Wörter</th>
                            <th>Code (10-stellige Hex)</th>
                            <th>Aufruf (copy)</th>
                            <th>Aktionen</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($clouds as $cloud) : ?>
                        <tr>
                            <td><?php echo esc_html($cloud->title); ?></td>
                            <td><?php echo intval($this->get_word_count_for_cloud($cloud->id)); ?></td>
                            <td><code><?php echo esc_html($cloud->code); ?></code></td>
                            <td><?php echo '<input type="text" class="copy-to-clipboard" style="direction:rtl" value="'.home_url().'/wordcloud/?code='.esc_html($cloud->code).'">'; ?></td>
                            <td>
                                <a class="button" href="<?php
                                    echo esc_url(wp_nonce_url(
                                        add_query_arg(
                                            ['page' => 'menti-clouds', 'menti_action' => 'export', 'cloud_id' => intval($cloud->id)],
                                            admin_url('admin.php')
                                        ),
                                        'menti_export_cloud_' . intval($cloud->id)
                                    ));
                                ?>">Export CSV</a>

                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="menti_action" value="reset" />
                                    <input type="hidden" name="cloud_id" value="<?php echo intval($cloud->id); ?>" />
                                    <?php wp_nonce_field('menti_reset_cloud_' . intval($cloud->id)); ?>
                                    <?php submit_button('Reset', 'secondary', '', false, [
                                        'onclick' => "return confirm('Alle Wörter für diese Wortwolke wirklich löschen?');"
                                    ]); ?>
                                </form>

                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="menti_action" value="delete" />
                                    <input type="hidden" name="cloud_id" value="<?php echo intval($cloud->id); ?>" />
                                    <?php wp_nonce_field('menti_delete_cloud_' . intval($cloud->id)); ?>
                                    <?php submit_button('Löschen', 'delete', '', false, [
                                        'onclick' => "return confirm('Diese Wortwolke UND alle dazugehörigen Wörter wirklich löschen?');"
                                    ]); ?>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p>Noch keine Wortwolken angelegt.</p>
            <?php endif; ?>
        </div>
        <?php
    }

    /** Summe aller count-Werte */
    private function get_total_mentions_for_cloud($cloud_id) {
        global $wpdb;
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(count) FROM {$this->table_words} WHERE cloud_id = %d",
            $cloud_id
        ));
    }

    /**
     * Statistik + Zeitraum für eine Cloud (für Footer + Live-Update per AJAX)
     * @return array{word_count:int,total_mentions:int,min:float,max:float,avg:float,min_fmt:string,max_fmt:string,avg_fmt:string,footer_html:string,top_html:string,timespan_html:string}
     */
    private function get_stats_for_cloud($cloud_id, $words = null, $total_mentions = null) {
        if (!is_array($words)) {
            $words = $this->get_words_for_cloud($cloud_id);
        }
        if ($total_mentions === null) {
            $total_mentions = $this->get_total_mentions_for_cloud($cloud_id);
        }
        $word_count = is_array($words) ? count($words) : 0;

        // Prozentwerte
        $percents = [];
        if (is_array($words) && $total_mentions > 0) {
            foreach ($words as $row) {
                $count = (int) ($row['count'] ?? 0);
                $percents[] = ($count / $total_mentions) * 100;
            }
        }
        $min_p = $percents ? min($percents) : 0;
        $max_p = $percents ? max($percents) : 0;
        $avg_p = $percents ? (array_sum($percents) / count($percents)) : 0;

        // Zeitraum
        $oldest_ts = null;
        $newest_ts = null;
        if (is_array($words) && !empty($words)) {
            foreach ($words as $row) {
                $dt_value = $row['created_at'] ?? '';
                $ts = is_numeric($dt_value) ? (int)$dt_value : strtotime((string)$dt_value);
                if (!$ts) continue;

                if ($newest_ts === null || $ts > $newest_ts) $newest_ts = $ts;
                if ($oldest_ts === null || $ts < $oldest_ts) $oldest_ts = $ts;
            }
        }

        $timespan_html = '';
        if ($oldest_ts && $newest_ts) {
            $oldest_dt = new DateTime('@' . $oldest_ts);
            $newest_dt = new DateTime('@' . $newest_ts);
            $oldest_dt->setTimezone(wp_timezone());
            $newest_dt->setTimezone(wp_timezone());
            $diff = $oldest_dt->diff($newest_dt);

            $timespan_html =
                ' | Zeitraum: <b>' . esc_html($oldest_dt->format('d.m.Y H:i')) . '</b>' .
                ' – <b>' . esc_html($newest_dt->format('d.m.Y H:i')) . '</b>' .
                ' (⏱️ ' . intval($diff->days) . ' Tage)';
        }

        $min_fmt = number_format($min_p, 1, ',', '') . ' %';
        $max_fmt = number_format($max_p, 1, ',', '') . ' %';
        $avg_fmt = number_format($avg_p, 1, ',', '') . ' %';

        $footer_html =
            '<b>Gesamt</b> - 📜 Beiträge: ' . ($word_count) .
            ' - 👍 Upvotes: ' . intval($total_mentions) .
            ' - 📉 Min %: ' . number_format($min_p, 1, ',', '') .
            ' - 📈 Max %: ' . number_format($max_p, 1, ',', '') .
            ' - Ø %: ' . number_format($avg_p, 1, ',', '') .
            $timespan_html;

        $top_html =
            '📜 Beiträge: <b>' . ($word_count) . '</b>' .
            ' · 👍 Upvotes: <b>' . intval($total_mentions) . '</b>' .
            ' · 📉 Min: <b>' . esc_html($min_fmt) . '</b>' .
            ' · Ø: <b>' . esc_html($avg_fmt) . '</b>' .
            ' · 📈 Max: <b>' . esc_html($max_fmt) . '</b>' .
            $timespan_html;

        return [
            'word_count' => (int)$word_count,
            'total_mentions' => (int)$total_mentions,
            'min' => (float)$min_p,
            'max' => (float)$max_p,
            'avg' => (float)$avg_p,
            'min_fmt' => $min_fmt,
            'max_fmt' => $max_fmt,
            'avg_fmt' => $avg_fmt,
            'timespan_html' => $timespan_html,
            'footer_html' => $footer_html,
            'top_html' => $top_html,
        ];
    }


    private function is_valid_code($code) {
        return $code !== '' && preg_match(self::CODE_RE, $code);
    }

    private function get_cloud_by_code($code) {
        global $wpdb;
        if (!$this->is_valid_code($code)) return null;

        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_clouds} WHERE code = %s",
            strtolower($code)
        ));
    }

    private function pccf($content) {
        return function_exists('pccf_filter') ? pccf_filter($content) : $content;
    }

    /** Shortcode [menti_cloud] */
    public function shortcode_menti_cloud($atts) {
        global $wpdb;
        $code = isset($_GET['code']) ? sanitize_text_field(wp_unslash($_GET['code'])) : '';
        if ($code === '') return '<p>Kein Code angegeben. Bitte rufe die Seite mit <code>?code=DEINCODE</code> auf.</p>';
        if (!$this->is_valid_code($code)) return '<p>Ungültiger Code.</p>';
        $cloud = $this->get_cloud_by_code($code);
        if (!$cloud) return '<p>Es wurde keine Wortwolke für diesen Code gefunden.</p>';
        $words = $this->get_words_for_cloud($cloud->id);
		// Zeitraum (ältester bis neuester Beitrag) aus $words ermitteln
		$oldest_ts = null;
		$newest_ts = null;
		if (!empty($words)) {
			foreach ($words as $row) {
				$dt_value = $row['created_at'] ?? '';
				$ts = is_numeric($dt_value) ? (int)$dt_value : strtotime((string)$dt_value);
				if (!$ts) continue;

				if ($newest_ts === null || $ts > $newest_ts) $newest_ts = $ts;
				if ($oldest_ts === null || $ts < $oldest_ts) $oldest_ts = $ts;
			}
		}
		$timespan_text = '';
		if ($oldest_ts && $newest_ts) {
			$oldest_dt = new DateTime('@' . $oldest_ts);
			$newest_dt = new DateTime('@' . $newest_ts);
			$oldest_dt->setTimezone(wp_timezone());
			$newest_dt->setTimezone(wp_timezone());

			$diff = $oldest_dt->diff($newest_dt);

			$timespan_text =
				' | Zeitraum: <b>' . esc_html($oldest_dt->format('d.m.Y H:i')) . '</b>' .
				' – <b>' . esc_html($newest_dt->format('d.m.Y H:i')) . '</b>' .
				' (⏱️ ' . intval($diff->days) . ' Tage)';
		}
        $word_count = is_array($words) ? count($words) : 0;
        $total_mentions = $this->get_total_mentions_for_cloud($cloud->id);
        $stats = $this->get_stats_for_cloud($cloud->id, $words, $total_mentions);
        // Frontend-Skript nur laden, wenn der Shortcode wirklich genutzt wird
        $this->register_frontend_script();
        wp_enqueue_script('mm-cloud');
        wp_localize_script('mm-cloud', 'mentiCloud', [
            'ajax_url'      => admin_url('admin-ajax.php'),
            'nonce'         => wp_create_nonce('menti_cloud_nonce'),
            'code'          => $cloud->code,
            'initial_words' => array_values($words),

            // Delete nur für Admins im Frontend
            'can_delete'    => current_user_can('manage_options'),
            'delete_nonce'  => wp_create_nonce('menti_delete_word_nonce'),
        ]);

        $tcolor = get_theme_mod('link-color', '#006060');

        ob_start();
        ?>
		<style>
		#menti-cloud-wrapper{border:1px solid <?php echo $tcolor;?>;padding:4px;background:<?php echo $tcolor.'15';?>;max-width:100%;margin:4px auto}
		#menti-cloud-words{min-height:60px;padding:10px;display:flex;flex-wrap:wrap;gap:10px}
		.menti-item{background:#fffb;border:1px solid #0002;padding:8px 10px;flex:1 1 280px;min-width:240px;box-shadow:0 1px 2px #0001;cursor:pointer}
		.menti-item:hover{filter:brightness(.98)}
		.menti-item-header{display:flex;align-items:center;justify-content:space-between;gap:10px;font-size:.85em;opacity:.85;margin-bottom:6px}
		.menti-item-left{display:inline-flex;align-items:center;gap:8px}
		.menti-item-body{font-size:1.05em;line-height:1.3}
		.menti-item-upvotes{white-space:nowrap}
		.menti-pill{display:inline-block;padding:2px 8px;background:#0001}
		.menti-delete{cursor:pointer}
		.menti-delete:hover{filter:brightness(0.8)}
		.menti-topbar{display:flex;align-items:flex-start;justify-content:space-between;gap:6px;flex-wrap:wrap}
		.menti-topbar-left{flex:1 1 360px}
		.menti-topbar-right{flex:0 0 auto}
		.menti-cloud-table th:nth-child(3),.menti-cloud-table td:nth-child(3){width:42%}
		.menti-cloud-stats{display:flex;flex-wrap:wrap;gap:12px;align-items:center;justify-content:center;font-size:.9em}
		.menti-cloud-stats b{font-weight:700}
		@media (max-width:599px) {
		.hide-on-portrait-small{display:none!important}
		}
		@media (orientation:portrait) {
		.hide-on-portrait-small{display:none!important}
		}
		@media print { .noprint { display: none !important; }  @page { size: landscape; } }
		</style>
        <div id="menti-cloud-wrapper">
            <h2 class="widget-title"><?php echo '📚 ' .esc_html($this->pccf($cloud->title)); ?></h2>
            <div id="menti-cloud" data-code="<?php echo esc_attr($cloud->code); ?>">
                <div class="menti-topbar noprint">
                    <div class="menti-topbar-left">
						<div id="menti-cloud-stats-top" class="menti-cloud-stats menti-cloud-stats-top"><?php echo $stats['top_html']; ?></div>
						<form id="menti-cloud-form" class="noprint" method="post" style="background:#fffa;text-align:center">
                            <label for="menti-word-input">Neues Wort oder Satz eintragen: (Upvote: Jeder kann vorhandene Beiträge anklicken, um zuzustimmen)</label><br>
                            <input type="text" id="menti-author-input" name="author" maxlength="3" style="width:80px; text-transform:uppercase;" placeholder="Kürzel" aria-label="Autor-Kürzel (3 Zeichen)">
                            <input type="text" id="menti-word-input" placeholder="Dein Beitrag" name="word" maxlength="255" required>
                            <button type="submit">Senden</button>
                            <span class="menti-inline-controls">
                                <label>
                                    🔀<span class="screen-reader-text">Sortierung</span>
                                    <select id="menti-sort" aria-label="Sortierung">
                                        <option value="newest" selected>Neueste zuerst</option>
                                        <option value="top">Meiste Upvotes zuerst</option>
                                    </select>
                                </label>
                                <label>
                                    🏷️<span class="screen-reader-text">Filter Kürzel</span>
                                    <select id="menti-filter-author" aria-label="Filter nach Kürzel">
                                        <option value="ALL" selected>Alle</option>
                                    </select>
                                </label>
                            </span>

                        </form>
                    </div>
                    <div class="menti-topbar-right hide-on-portrait-small" title="mit dem Smartphone scannen">
                        <?php
                        global $wp;
                        $current_url_with_args = home_url(add_query_arg($_GET, $wp->request));
                        if (shortcode_exists('qrcode')) echo do_shortcode('[qrcode text='.$current_url_with_args.' size=2]');
                        ?>
                    </div>
                </div>

                <div id="menti-cloud-words"></div>
            </div>
        </div>
        <?php
        if (!empty($words) && $total_mentions > 0) :
            usort($words, fn($a, $b) => ($b['count'] ?? 0) <=> ($a['count'] ?? 0));
            ?>
            
			<div id="menti-cloud-table-wrapper">
				<h6>📊 Auswertung</h6>

				<span style="float:right">
					<a class="button button-secondary" href="<?php
						echo esc_url(wp_nonce_url(
							add_query_arg(
								['page' => 'menti-clouds', 'menti_action' => 'export', 'cloud_id' => intval($cloud->id)],
								admin_url('admin.php')
							),
							'menti_export_cloud_' . intval($cloud->id)
						));
					?>">CSV Export (Tabelle)</a>
				</span>
                <table class="menti-cloud-table">
                    <thead>
                    <tr>
                        <th>Datum/Zeit</th>
                        <th>Autor</th>
                        <th>Titel</th>
                        <th>Nennungen</th>
                        <th>Anteil (%)</th>
                        <th>Graph</th>
                    </tr>
                    </thead>
                    <tbody id="menti-cloud-table-body">
                    <?php foreach ($words as $row) :
                        $count = (int) ($row['count'] ?? 0);
                        $percent = $total_mentions > 0 ? ($count / $total_mentions) * 100 : 0;
                        ?>
                        <tr>
                            <td><?php
                                $dt_value = $row['created_at'] ?? '';
                                $dt_ts = is_numeric($dt_value) ? (int)$dt_value : strtotime((string)$dt_value);
                                if (function_exists('colordatebox') && $dt_ts) echo colordatebox($dt_ts, NULL, NULL, 1);
                                else echo esc_html($this->format_datetime($dt_value));
                                ?></td>
                            <td><?php echo esc_html(trim(($row['emoji'] ?? '') . ' ' . ($row['author'] ?? ''))); ?></td>
                            <td><?php echo esc_html($this->pccf($row['word'] ?? '')); ?></td>
                            <td><?php echo $count; ?></td>
                            <td><?php echo number_format($percent, 1, ',', ''); ?></td>
                            <td><?php echo '<progress class="progressleer" style="min-width:200px;width:100%" max="100" value="'.$percent.'"></progress>'; ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
					<tfoot>
					  <tr>
						<td colspan="6">
						  <?php echo '<span id="menti-cloud-stats-footer">' . $stats['footer_html'] . '</span>'; ?>
						</td>
					  </tr>
					</tfoot>
                </table>
            </div>
        <?php endif;

        return ob_get_clean();
    }

    /** Wortliste für eine Cloud */
    private function get_words_for_cloud($cloud_id) {
        global $wpdb;
        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT word, author, emoji, count, created_at FROM {$this->table_words} WHERE cloud_id = %d ORDER BY created_at DESC",
            $cloud_id
        ), ARRAY_A);
        return $rows ?: [];
    }

    /** Autor-Kürzel: genau 3 Zeichen (A-Z/0-9), sonst XXX */
    private function sanitize_author_initials($raw) {
        $raw = preg_replace('/[^A-Z0-9]/', '', strtoupper(trim((string)$raw)));
        return (strlen($raw) === 3) ? $raw : 'XXX';
    }

    /** Pro Cloud: Kürzel -> Emoji (persistiert) */
    private function get_or_assign_author_emoji($cloud_id, $author) {
        $option_key = 'menti_emoji_map_' . (int)$cloud_id;
        $map = get_option($option_key, []);
        if (!is_array($map)) $map = [];


        // Fallback: leeres/ungültiges Kürzel wird als XXX behandelt und bekommt immer 🕵
        if ($author === '' || $author === 'XXX') {
            $map['XXX'] = '🕵';
            update_option($option_key, $map, false);
            return '🕵';
        }

        // Migration: falls XXX historisch auf 🍌 oder anderes zeigte, korrigieren wir auf 🕵
        if (!empty($map['XXX']) && $map['XXX'] !== '🕵') {
            $map['XXX'] = '🕵';
            update_option($option_key, $map, false);
        }
        if (!empty($map[$author]) && is_string($map[$author])) return $map[$author];

        $pool = self::get_emoji_pool();
        $used = array_values($map);
        $chosen = '';

        if ($pool) {
            $start = abs(crc32((string)$author)) % count($pool);
            $n = count($pool);
            for ($i = 0; $i < $n; $i++) {
                $candidate = $pool[($start + $i) % $n];
                if (!in_array($candidate, $used, true)) { $chosen = $candidate; break; }
            }
            if ($chosen === '') $chosen = $pool[$start];
        }

        $map[$author] = $chosen ?: '🕵';
        update_option($option_key, $map, false);
        return $map[$author];
    }

    /** Emoji-Pool (static, damit nicht jedes Mal neu gebaut) */
    private static function get_emoji_pool() {
        static $pool;
        if ($pool) return $pool;
        return $pool = [
			'🙂','😊','😄','😁','😉','😎','🤓','🥳', '🔴','🟠','🟡','🟢','🔵','🟣', '🔶','🔷',
			'🔥','💧','❄️','⚡','🌈','🍀','🌸','🌻','🍁','🌍','🌙','☀️','🌧️','🌤️','🌪️',
			'🍎','🍊','🍋','🍇','🍉','🥝','🍓','🫐','🍒','🍑','🍍','🥥','🥕','🌽','🥑','🍔','🍕',
			'⚽','🏀','🏈','🏓','🎯','🥏', '🎈','🎉','🎁','🧩','🎨','🧵','🧶',
			'📌','📦','📚','📘','📕','📗','📙','📒', '💡','🔔','⏰','🧭','🔑','🗝️','🧲',
			'💻','🖥️','📱','📺','📷','🎥','🎮','📻','🖨️',
			'🚗','🚙','🚕','🚌','🚲','🏍️','✈️','🚀','🚁','🚢','⛵','🚤',
			'🏠','🏢','🏫','🏭','🏬','🏥','🏦','❤️','💙','💚','💛','💜','🧡','🤎'
		];
    }

    /** MySQL DATETIME -> lokales WP Datum/Zeit Format */
    private function format_datetime($mysql_datetime) {
        $mysql_datetime = (string)$mysql_datetime;
        if ($mysql_datetime === '') return '';

        $ts = strtotime($mysql_datetime . ' UTC') ?: strtotime($mysql_datetime);
        if (!$ts) return $mysql_datetime;

        return function_exists('wp_date') ? wp_date('d.m.Y H:i', $ts) : date_i18n('d.m.Y H:i', $ts);
    }

    /** Ajax: neues Wort (oder bestehendes erhöhen) */
    public function ajax_add_word() {
        check_ajax_referer('menti_cloud_nonce', 'nonce');

        $code = isset($_POST['code']) ? sanitize_text_field(wp_unslash($_POST['code'])) : '';
        $word = isset($_POST['word']) ? sanitize_text_field(wp_unslash($_POST['word'])) : '';
        $author = $this->sanitize_author_initials(isset($_POST['author']) ? sanitize_text_field(wp_unslash($_POST['author'])) : '');

        if ($code === '' || $word === '') wp_send_json_error(['message' => 'Code oder Wort fehlt.']);
        if (!$this->is_valid_code($code)) wp_send_json_error(['message' => 'Ungültiger Code.']);

        $word = trim($word);
        if (strlen($word) < 1 || strlen($word) > 255) wp_send_json_error(['message' => 'Der Text muss zwischen 1 und 255 Zeichen haben.']);

        global $wpdb;
        $cloud = $this->get_cloud_by_code($code);
        if (!$cloud) wp_send_json_error(['message' => 'Wortwolke nicht gefunden.']);

        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_words} WHERE cloud_id = %d AND word = %s",
            $cloud->id, $word
        ));

        if ($existing) {
            $wpdb->update($this->table_words, ['count' => $existing->count + 1], ['id' => $existing->id], ['%d'], ['%d']);
        } else {
            $emoji = $this->get_or_assign_author_emoji($cloud->id, $author);
            $wpdb->insert($this->table_words, [
                'cloud_id' => $cloud->id,
                'word'     => $this->pccf($word),
                'author'   => $author,
                'emoji'    => $emoji,
                'count'    => 1,
            ], ['%d','%s','%s','%s','%d']);
        }

        $this->mark_word_as_voted($code, $word);

        wp_send_json_success([
            'message' => 'Eintrag gespeichert.',
            'words'   => $this->get_words_for_cloud($cloud->id),
            'stats'   => $this->get_stats_for_cloud($cloud->id),
        ]);
    }

    /** Ajax: Wort klicken (Vote +1, aber pro Wort & Browser nur einmal) */
    public function ajax_vote_word() {
        check_ajax_referer('menti_cloud_nonce', 'nonce');

        $code = isset($_POST['code']) ? sanitize_text_field(wp_unslash($_POST['code'])) : '';
        $word = isset($_POST['word']) ? sanitize_text_field(wp_unslash($_POST['word'])) : '';

        if ($code === '' || $word === '') wp_send_json_error(['message' => 'Code oder Wort fehlt.']);
        if (!$this->is_valid_code($code)) wp_send_json_error(['message' => 'Ungültiger Code.']);

        $word = trim($word);
        if ($this->has_voted_for_word($code, $word)) wp_send_json_error(['message' => 'Sie haben diesen Eintrag bereits unterstützt.']);

        global $wpdb;
        $cloud = $this->get_cloud_by_code($code);
        if (!$cloud) wp_send_json_error(['message' => 'Wortwolke nicht gefunden.']);

        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_words} WHERE cloud_id = %d AND word = %s",
            $cloud->id, $word
        ));
        if (!$existing) wp_send_json_error(['message' => 'Dieser Eintrag existiert nicht (mehr).']);

        $wpdb->update($this->table_words, ['count' => $existing->count + 1], ['id' => $existing->id], ['%d'], ['%d']);
        $this->mark_word_as_voted($code, $word);

        wp_send_json_success([
            'message' => 'Danke für Ihre Unterstützung.',
            'words'   => $this->get_words_for_cloud($cloud->id),
            'stats'   => $this->get_stats_for_cloud($cloud->id),
        ]);
    }

    /** Cookie-Handling */
    private function has_voted_for_word($code, $word) {
        $cookie = 'menti_voted_' . strtolower($code);
        if (empty($_COOKIE[$cookie])) return false;

        $data = json_decode(wp_unslash($_COOKIE[$cookie]), true);
        return is_array($data) && in_array($word, $data, true);
    }

    private function mark_word_as_voted($code, $word) {
        $cookie = 'menti_voted_' . strtolower($code);
        $data = [];

        if (!empty($_COOKIE[$cookie])) {
            $stored = json_decode(wp_unslash($_COOKIE[$cookie]), true);
            if (is_array($stored)) $data = $stored;
        }

        if (!in_array($word, $data, true)) $data[] = $word;

        setcookie(
            $cookie,
            wp_json_encode($data),
            time() + YEAR_IN_SECONDS,
            COOKIEPATH ? COOKIEPATH : '/',
            COOKIE_DOMAIN ? COOKIE_DOMAIN : ''
        );
    }

    /** Farbhelper */

    /** Ajax: Wort/Beitrag löschen (nur Admin im Frontend) */
    public function ajax_delete_word() {
        // eigener Nonce (separat von menti_cloud_nonce)
        check_ajax_referer('menti_delete_word_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung.']);
        }

        $code = isset($_POST['code']) ? sanitize_text_field(wp_unslash($_POST['code'])) : '';
        $word = isset($_POST['word']) ? sanitize_text_field(wp_unslash($_POST['word'])) : '';

        if ($code === '' || $word === '') {
            wp_send_json_error(['message' => 'Code oder Wort fehlt.']);
        }
        if (!$this->is_valid_code($code)) {
            wp_send_json_error(['message' => 'Ungültiger Code.']);
        }

        $word = trim($word);

        global $wpdb;
        $cloud = $this->get_cloud_by_code($code);
        if (!$cloud) {
            wp_send_json_error(['message' => 'Wortwolke nicht gefunden.']);
        }

        // Löschen über (cloud_id, word) – in der Tabelle ohnehin UNIQUE (cloud_word)
        $deleted = $wpdb->delete(
            $this->table_words,
            ['cloud_id' => (int) $cloud->id, 'word' => $word],
            ['%d', '%s']
        );

        if ($deleted === false) {
            wp_send_json_error(['message' => 'DB-Fehler beim Löschen.']);
        }

        wp_send_json_success([
            'message' => 'Eintrag gelöscht.',
            'words'   => $this->get_words_for_cloud($cloud->id),
            'stats'   => $this->get_stats_for_cloud($cloud->id),
        ]);
    }

    public function mm_tagcolor_luminance($hex, $percent) {
        if (empty($hex)) $hex = get_theme_mod('link-color', '#666666');

        $hex = preg_replace('/[^0-9a-f]/i', '', $hex);
        if (strlen($hex) === 3) $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        if (strlen($hex) !== 6) $hex = '666666';

        $new = '#';
        for ($i = 0; $i < 3; $i++) {
            $dec = hexdec(substr($hex, $i * 2, 2));
            $dec = (int) min(max(0, $dec + $dec * $percent), 255);
            $new .= str_pad(dechex($dec), 2, '0', STR_PAD_LEFT);
        }
        return $new;
    }

    private function get_tag_colors() {
        $base = get_theme_mod('link-color', '#666666');
        $colors = [];
        foreach (self::TAGCOLOR_STEPS as $p) $colors[] = $this->mm_tagcolor_luminance($base, $p);
        return $colors ?: ['#1abc9c','#3498db','#9b59b6','#e67e22','#e74c3c','#2ecc71'];
    }

    /** Frontend-Script registrieren (inline JS) */
    /** Ajax: Tabelle + Statistik passend zur aktuellen Sortierung/Filter zurückgeben */
    public function ajax_cloud_view() {
        check_ajax_referer('menti_cloud_nonce', 'nonce');

        $code  = isset($_POST['code']) ? sanitize_text_field(wp_unslash($_POST['code'])) : '';
        $sort  = isset($_POST['sort']) ? sanitize_text_field(wp_unslash($_POST['sort'])) : 'newest';
        $author = isset($_POST['author']) ? sanitize_text_field(wp_unslash($_POST['author'])) : 'ALL';

        if ($code === '' || !$this->is_valid_code($code)) {
            wp_send_json_error(['message' => 'Ungültiger Code.']);
        }

        $cloud = $this->get_cloud_by_code($code);
        if (!$cloud) {
            wp_send_json_error(['message' => 'Wortwolke nicht gefunden.']);
        }

        $words = $this->get_words_for_cloud($cloud->id);
        if (!is_array($words)) $words = [];

        // Filter (Autor)
        if ($author && $author !== 'ALL') {
            $words = array_values(array_filter($words, function ($w) use ($author) {
                $a = strtoupper(trim((string)($w['author'] ?? '')));
                return $a === strtoupper(trim((string)$author));
            }));
        }

        // Sortierung (wie im JS)
        if ($sort === 'top') {
            usort($words, function ($a, $b) {
                $ca = (int)($a['count'] ?? 0);
                $cb = (int)($b['count'] ?? 0);
                if ($cb !== $ca) return $cb <=> $ca;

                $ta = strtotime((string)($a['created_at'] ?? '')) ?: 0;
                $tb = strtotime((string)($b['created_at'] ?? '')) ?: 0;
                return $tb <=> $ta;
            });
        } else {
            usort($words, function ($a, $b) {
                $ta = strtotime((string)($a['created_at'] ?? '')) ?: 0;
                $tb = strtotime((string)($b['created_at'] ?? '')) ?: 0;
                return $tb <=> $ta;
            });
        }

        // Total Mentions passend zur gefilterten Ansicht (damit Prozent/Progress identisch “stimmt”)
        $total_mentions = 0;
        foreach ($words as $row) {
            $total_mentions += (int)($row['count'] ?? 0);
        }

        $stats = $this->get_stats_for_cloud($cloud->id, $words, $total_mentions);

        // tbody HTML exakt wie im Shortcode
        ob_start();
        foreach ($words as $row) :
            $count = (int) ($row['count'] ?? 0);
            $percent = $total_mentions > 0 ? ($count / $total_mentions) * 100 : 0;
            ?>
            <tr>
                <td><?php
                    $dt_value = $row['created_at'] ?? '';
                    $dt_ts = is_numeric($dt_value) ? (int)$dt_value : strtotime((string)$dt_value);
                    if (function_exists('colordatebox') && $dt_ts) echo colordatebox($dt_ts, NULL, NULL, 1);
                    else echo esc_html($this->format_datetime($dt_value));
                ?></td>
                <td><?php echo esc_html(trim(($row['emoji'] ?? '') . ' ' . ($row['author'] ?? ''))); ?></td>
                <td><?php echo esc_html($this->pccf($row['word'] ?? '')); ?></td>
                <td><?php echo $count; ?></td>
                <td><?php echo number_format($percent, 1, ',', ''); ?></td>
                <td><?php echo '<progress class="progress-bar" style="width:100%" max="100" value="'.$percent.'"></progress>'; ?></td>
            </tr>
            <?php
        endforeach;
        $tbody_html = ob_get_clean();

        wp_send_json_success([
            'tbody_html' => $tbody_html,
            'stats'      => $stats,
            'total_mentions' => $total_mentions,
            'items'      => count($words),
        ]);
    }
    public function register_frontend_script() {
        wp_register_script('mm-cloud', false, [], '1.2.0', true);
        $tagcolors = wp_json_encode($this->get_tag_colors());

		$inline_js = <<<JS
		(function () {
		  'use strict';
		  var allWords = [];
		  var currentSort = 'newest';
		  var currentAuthor = 'ALL';
		  function $(id) { return document.getElementById(id); }
		  function toInt(v) { v = parseInt(v, 10); return isNaN(v) ? 0 : v; }
		  function parseMysqlDate(mysqlDateTime) {
			if (!mysqlDateTime) return 0;
			var iso = String(mysqlDateTime).replace(' ', 'T') + 'Z';
			var d = new Date(iso);
			if (isNaN(d.getTime())) d = new Date(String(mysqlDateTime).replace(' ', 'T'));
			return isNaN(d.getTime()) ? 0 : d.getTime();
		  }
		  function formatDate(mysqlDateTime) {
			if (!mysqlDateTime) return '';
			var iso = String(mysqlDateTime).replace(' ', 'T') + 'Z';
			var d = new Date(iso);
			if (isNaN(d.getTime())) d = new Date(String(mysqlDateTime).replace(' ', 'T'));
			if (isNaN(d.getTime())) return mysqlDateTime;
			return d.toLocaleString('de-DE', {
			  year: 'numeric', month: '2-digit', day: '2-digit',
			  hour: '2-digit', minute: '2-digit'
			});
		  }
		  function authorKey(w) { return ((w && w.author) ? String(w.author) : 'XXX').trim(); }
		  function emoji(w) { return ((w && w.emoji) ? String(w.emoji) : '').trim(); }
		  function postAjax(url, formData) {
			return fetch(url, {
			  method: 'POST',
			  body: formData,
			  credentials: 'same-origin'
			}).then(function (r) { return r.json(); });
		  }

		  function applySortAndFilter(words) {
			var out = Array.isArray(words) ? words.slice() : [];

			if (currentAuthor && currentAuthor !== 'ALL') {
			  out = out.filter(function (w) { return authorKey(w) === currentAuthor; });
			}

			if (currentSort === 'top') {
			  out.sort(function (a, b) {
				var ca = toInt(a.count), cb = toInt(b.count);
				if (cb !== ca) return cb - ca;
				return parseMysqlDate(b.created_at) - parseMysqlDate(a.created_at);
			  });
			} else {
			  out.sort(function (a, b) {
				return parseMysqlDate(b.created_at) - parseMysqlDate(a.created_at);
			  });
			}
			return out;
		  }

		  function populateAuthorFilter(words) {
			var sel = $('menti-filter-author');
			if (!sel) return;

			var previous = sel.value || currentAuthor || 'ALL';
			var map = {};

			(Array.isArray(words) ? words : []).forEach(function (w) {
			  var k = authorKey(w) || 'XXX';
			  map[k] = emoji(w);
			});

			var keys = Object.keys(map).sort();
			sel.innerHTML = '';

			var optAll = document.createElement('option');
			optAll.value = 'ALL';
			optAll.textContent = 'Alle';
			sel.appendChild(optAll);

			keys.forEach(function (k) {
			  var opt = document.createElement('option');
			  opt.value = k;
			  opt.textContent = (map[k] ? (map[k] + ' ') : '') + k;
			  sel.appendChild(opt);
			});

			var exists = Array.prototype.some.call(sel.options, function (o) { return o.value === previous; });
			sel.value = exists ? previous : 'ALL';
			currentAuthor = sel.value;
		  }

		  function renderCloud(words) {
			var container = $('menti-cloud-words');
			if (!container) return;

			container.innerHTML = '';

			if (!Array.isArray(words) || words.length === 0) {
			  container.textContent = (Array.isArray(allWords) && allWords.length > 0)
				? 'Keine Einträge für den aktuellen Filter.'
				: 'Noch keine Einträge.';
			  return;
			}

			var counts = words.map(function (w) { return toInt(w.count); });
			var min = Math.min.apply(null, counts);
			var max = Math.max.apply(null, counts);

			var minSize = 13, maxSize = 24;
			var colors = $tagcolors; // aus PHP eingesetzt

			words.forEach(function (w) {
			  var item = document.createElement('div');
			  item.className = 'menti-item';
				item.title = 'Klicke für Deine Zustimmung';
				
			  var c = toInt(w.count);

			  var size = (min === max)
				? (minSize + maxSize) / 2
				: (minSize + ((c - min) / (max - min)) * (maxSize - minSize));

			  var colorIndex;
			  if (min === max) {
				colorIndex = Math.floor(colors.length / 2);
			  } else {
				var ratio = (c - min) / (max - min);
				colorIndex = Math.round(ratio * (colors.length - 1));
			  }
			  colorIndex = Math.max(0, Math.min(colors.length - 1, colorIndex));

			  item.style.borderLeft = '6px solid ' + colors[colorIndex];
			  item.dataset.word = w.word;

			  // Header
			  var header = document.createElement('div');
			  header.className = 'menti-item-header';

			  var left = document.createElement('div');
			  left.className = 'menti-item-left';

			  var dt = document.createElement('span');
			  dt.className = 'menti-pill';
			  dt.textContent = formatDate(w.created_at);

			  var au = document.createElement('span');
			  au.className = 'menti-pill';
			  au.textContent = (emoji(w) ? (emoji(w) + ' ') : '') + authorKey(w);

			  left.appendChild(dt);
			  left.appendChild(au);

			  var up = document.createElement('span');
			  up.className = 'menti-item-upvotes menti-pill';
			  up.textContent = '👍 ' + c;

			  // Admin: Delete-Icon (🗑️) im Frontend
			  if (window.mentiCloud && mentiCloud.can_delete) {
				var del = document.createElement('span');
				del.className = 'fa fa-trash';
				del.title = 'Eintrag löschen';
				del.style.marginLeft = '8px';

				del.addEventListener('click', function (ev) {
				  ev.preventDefault();
				  ev.stopPropagation(); // verhindert Vote über Container-Click

				  if (!confirm('Diesen Eintrag wirklich löschen?')) return;

				  deleteWord(mentiCloud.code, w.word, mentiCloud.ajax_url, mentiCloud.delete_nonce);
				});

				up.appendChild(del);
			  }

			  header.appendChild(left);
			  header.appendChild(up);

			  // Body
			  var body = document.createElement('div');
			  body.className = 'menti-item-body';
			  body.style.fontSize = size + 'px';
			  body.textContent = w.word;

			  item.appendChild(header);
			  item.appendChild(body);
			  container.appendChild(item);
			});
		  }



		  function refreshTableView() {
			if (typeof mentiCloud === 'undefined') return;
			var tbody = document.getElementById('menti-cloud-table-body');
			if (!tbody) return;

			var ajaxUrl = mentiCloud.ajax_url;
			var nonce = mentiCloud.nonce;
			var code = mentiCloud.code;

			var fd = new FormData();
			fd.append('action', 'menti_cloud_view');
			fd.append('code', code);
			fd.append('sort', currentSort || 'newest');
			fd.append('author', currentAuthor || 'ALL');
			fd.append('nonce', nonce);

			postAjax(ajaxUrl, fd).then(function (res) {
				if (!res) return;
				if (res.success && res.data) {
					if (typeof res.data.tbody_html === 'string') tbody.innerHTML = res.data.tbody_html;

					if (res.data.stats) {
						var topEl = document.getElementById('menti-cloud-stats-top');
						if (topEl && res.data.stats.top_html) topEl.innerHTML = res.data.stats.top_html;

						var footEl = document.getElementById('menti-cloud-stats-footer');
						if (footEl && res.data.stats.footer_html) footEl.innerHTML = res.data.stats.footer_html;
					}
				}
			}).catch(function(){ /* still ignore */ });
		  }


		  function handleResponse(res) {
			if (!res) { alert('Es ist ein Fehler aufgetreten.'); return; }

			if (res.success) {
			  if (res.data && res.data.words) {
				allWords = res.data.words;
				populateAuthorFilter(allWords);
				renderCloud(applySortAndFilter(allWords));
				refreshTableView();
				refreshTableView();
				if (res.data && res.data.stats) {
					var topEl = document.getElementById('menti-cloud-stats-top');
					if (topEl && res.data.stats.top_html) topEl.innerHTML = res.data.stats.top_html;

					var footEl = document.getElementById('menti-cloud-stats-footer');
					if (footEl && res.data.stats.footer_html) footEl.innerHTML = res.data.stats.footer_html;
				}

			  }
			  if (res.data && res.data.message) console.log(res.data.message);
			} else {
			  alert((res.data && res.data.message) ? res.data.message : 'Es ist ein Fehler aufgetreten.');
			}
		  }
		  function voteWord(code, word, ajaxUrl, nonce) {
			var fd = new FormData();
			fd.append('action', 'menti_vote_word');
			fd.append('code', code);
			fd.append('word', word);
			fd.append('nonce', nonce);
			postAjax(ajaxUrl, fd).then(handleResponse).catch(function () {
			  alert('Netzwerkfehler beim Abstimmen.');
			});
		  }
		  function deleteWord(code, word, ajaxUrl, nonce) {
			var fd = new FormData();
			fd.append('action', 'menti_delete_word');
			fd.append('code', code);
			fd.append('word', word);
			fd.append('nonce', nonce);
			postAjax(ajaxUrl, fd).then(handleResponse).catch(function () {
			  alert('Netzwerkfehler beim Löschen.');
			});
		  }
		  function addWord(code, word, author, ajaxUrl, nonce) {
			var fd = new FormData();
			fd.append('action', 'menti_add_word');
			fd.append('code', code);
			fd.append('word', word);
			fd.append('author', author);
			fd.append('nonce', nonce);
			postAjax(ajaxUrl, fd).then(handleResponse).catch(function () {
			  alert('Netzwerkfehler beim Senden.');
			});
		  }
		  document.addEventListener('DOMContentLoaded', function () {
			if (typeof mentiCloud === 'undefined') return;
			var cloud = $('menti-cloud');
			if (!cloud) return;
			var sortSel = $('menti-sort');
			var authorSel = $('menti-filter-author');
			if (sortSel) {
			  currentSort = sortSel.value || 'newest';
			  sortSel.addEventListener('change', function () {
				currentSort = sortSel.value || 'newest';
				renderCloud(applySortAndFilter(allWords));
				refreshTableView();
			  });
			}
			if (authorSel) {
			  currentAuthor = authorSel.value || 'ALL';
			  authorSel.addEventListener('change', function () {
				currentAuthor = authorSel.value || 'ALL';
				renderCloud(applySortAndFilter(allWords));
				refreshTableView();
			  });
			}

			if (Array.isArray(mentiCloud.initial_words)) {
			  allWords = mentiCloud.initial_words;
			  populateAuthorFilter(allWords);
			  renderCloud(applySortAndFilter(allWords));
			}
			var code = mentiCloud.code;
			var ajaxUrl = mentiCloud.ajax_url;
			var nonce = mentiCloud.nonce;
			cloud.addEventListener('click', function (e) {
			  var el = e.target && e.target.closest ? e.target.closest('.menti-item') : null;
			  if (!el || !el.dataset || !el.dataset.word) return;
			  var word = el.dataset.word || '';
			  if (!word) return;
			  voteWord(code, word, ajaxUrl, nonce);
			});
			var form = $('menti-cloud-form');
			if (form) {
			  form.addEventListener('submit', function (e) {
				e.preventDefault();
				var input = form.querySelector('input[name="word"]');
				var authorInput = form.querySelector('input[name="author"]');
				if (!input || !input.value.trim()) return;
				var word = input.value.trim();
				var author = (authorInput && authorInput.value) ? authorInput.value.trim() : '';
				addWord(code, word, author, ajaxUrl, nonce);
				input.value = '';
				if (authorInput) authorInput.value = '';
			  });
			}
		  });
		})();
		JS;

        wp_add_inline_script('mm-cloud', $inline_js);
    }
}

// Hooks registrieren
register_activation_hook(__FILE__, ['MM_Wordcloud', 'activate']);

// Plugin initialisieren
MM_Wordcloud::instance();

// ----------------------------- Ende MM Wordcloud (Tagcloud) ähnlich wie Menti -----------------------------
